<?php
/*
 * Plugin Name: Infinity Pay 
 * Plugin URI: https://wordpress.org/plugins/infinitypay
 * Description: This plugin allows your customers to pay with Bkash, Nagad, Rocket, and all BD gateways via Infinity Pay.
 * Author: Infinity Digital Shop
 * Author URI: https://infinitydigitalshop.com
 * Version: 1.0.0
 * Requires at least: 5.2
 * Requires PHP: 7.2
 * License: GPL v2 or later
 * License URI: 
 * Text Domain: infinitypay
 */

/*
 * This action hook registers our PHP class as a WooCommerce payment gateway
 */
add_action('plugins_loaded', 'infinitypay_init_gateway_class');

function infinitypay_init_gateway_class()
{
    if (!class_exists('WC_Payment_Gateway')) return;

    class WC_InfinityPay_Gateway extends WC_Payment_Gateway
    {
        public function __construct()
        {
            $this->id = 'infinitypay';
            $this->icon = 'https://pay.infinitydigitalshop.com/infinitypay-icon.png';
            $this->has_fields = false;
            $this->method_title = __('Infinity Pay', 'infinitypay');
            $this->method_description = __('Pay With Infinity Pay - Secure payment gateway for Bangladesh', 'infinitypay');

            $this->supports = array('products');

            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->enabled = $this->get_option('enabled');
            $this->payment_site = $this->get_option('payment_site');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_api_' . strtolower(get_class($this)), array($this, 'handle_webhook'));
        }

        public function init_form_fields()
        {
            $this->form_fields = array(
                'enabled' => array(
                    'title'       => 'Enable/Disable',
                    'label'       => 'Enable Infinity Pay',
                    'type'        => 'checkbox',
                    'description' => '',
                    'default'     => 'no'
                ),
                'title' => array(
                    'title'       => 'Title',
                    'type'        => 'text',
                    'description' => 'This controls the title which the user sees during checkout.',
                    'default'     => 'Infinity Pay Gateway',
                    'desc_tip'    => true,
                ),
                'apikeys' => array(
                    'title'       => 'Enter API Key',
                    'type'        => 'text',
                    'description' => 'Get your API key from Infinity Pay dashboard',
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'currency_rate' => array(
                    'title'       => 'Enter USD Rate',
                    'type'        => 'number',
                    'description' => 'Conversion rate for USD to BDT',
                    'default'     => '110',
                    'desc_tip'    => true,
                ),
                'is_digital' => array(
                    'title'       => 'Enable/Disable Digital product',
                    'label'       => 'Enable Digital product',
                    'type'        => 'checkbox',
                    'description' => 'Mark orders as completed immediately for digital products',
                    'default'     => 'no'
                ),
                'payment_site' => array(
                    'title'       => 'Payment Site URL',
                    'type'        => 'text',
                    'description' => 'Infinity Pay payment gateway URL',
                    'default'     => 'https://pay.infinitydigitalshop.com/',
                    'desc_tip'    => true,
                ),
            );
        }

        public function process_payment($order_id)
        {
            global $woocommerce;
            $order = wc_get_order($order_id);
            $current_user = wp_get_current_user();

            // Calculate order total
            $subtotal = WC()->cart->subtotal;
            $shipping_total = WC()->cart->get_shipping_total();
            $fees = WC()->cart->get_fee_total();
            $discount_excl_tax_total = WC()->cart->get_cart_discount_total();
            $discount_tax_total = WC()->cart->get_cart_discount_tax_total();
            $discount_total = $discount_excl_tax_total + $discount_tax_total;
            $total = $subtotal + $shipping_total + $fees - $discount_total;

            // Convert to BDT if currency is USD
            if ($order->get_currency() == 'USD') {
                $total = $total * $this->get_option('currency_rate');
            }

            // Set initial order status
            if ($order->get_status() != 'completed') {
                $order->update_status('pending', __('Customer is being redirected to Infinity Pay', 'infinitypay'));
            }

            // Prepare payment data
            $data = array(
                "cus_name"    => $current_user->user_firstname . ' ' . $current_user->user_lastname,
                "cus_email"   => $current_user->user_email,
                "amount"      => round($total, 2),
                "webhook_url" => site_url('/?wc-api=wc_infinitypay_gateway&order_id=' . $order->get_id()),
                "success_url" => $this->get_return_url($order),
                "cancel_url"  => wc_get_checkout_url(),
                "order_id"    => $order->get_id()
            );

            $header = array(
                "api" => $this->get_option('apikeys'),
                "url" => rtrim($this->get_option('payment_site'), '/') . "/api/payment/create"
            );

            // Create payment request
            $response = $this->create_payment($data, $header);
            $data = json_decode($response, true);

            if (isset($data['payment_url'])) {
                return array(
                    'result'   => 'success',
                    'redirect' => $data['payment_url']
                );
            } else {
                $error_message = isset($data['message']) ? $data['message'] : __('Payment error: Could not connect to payment gateway.', 'infinitypay');
                wc_add_notice($error_message, 'error');
                $order->update_status('failed', $error_message);
                return;
            }
        }

        public function create_payment($data = "", $header = '')
        {
            $headers = array(
                'Content-Type: application/json',
                'API-KEY: ' . $header['api'],
            );
            $url = $header['url'];
            $curl = curl_init();
            $data = json_encode($data);

            curl_setopt_array($curl, array(
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_POSTFIELDS => $data,
                CURLOPT_HTTPHEADER => $headers,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_FAILONERROR => true
            ));

            $response = curl_exec($curl);
            
            if (curl_errno($curl)) {
                $error_msg = curl_error($curl);
                curl_close($curl);
                return json_encode(['error' => $error_msg]);
            }
            
            curl_close($curl);
            return $response;
        }

        public function update_order_status($order)
        {
            $transactionId = isset($_REQUEST['transactionId']) ? sanitize_text_field($_REQUEST['transactionId']) : '';
            
            if (empty($transactionId)) {
                $order->update_status('failed', __('Transaction ID not received from Infinity Pay', 'infinitypay'));
                return false;
            }

            $data = array(
                "transaction_id" => $transactionId,
            );
            
            $header = array(
                "api" => $this->get_option('apikeys'),
                "url" => rtrim($this->get_option('payment_site'), '/') . "/api/payment/verify"
            );

            $response = $this->create_payment($data, $header);
            $data = json_decode($response, true);

            if ($order->get_status() != 'completed') {
                if (isset($data['status']) && $data['status'] == "COMPLETED") {
                    $transaction_id = sanitize_text_field($data['transaction_id']);
                    $amount = isset($data['amount']) ? floatval($data['amount']) : 0;
                    $sender_number = isset($data['cus_email']) ? sanitize_email($data['cus_email']) : '';
                    $payment_method = isset($data['payment_method']) ? sanitize_text_field($data['payment_method']) : 'Infinity Pay';

                    $note = sprintf(
                        __("Infinity Pay payment successful. Payment Method: %s, Amount: %s, Transaction ID: %s, Sender: %s", 'infinitypay'),
                        $payment_method,
                        wc_price($amount),
                        $transaction_id,
                        $sender_number
                    );

                    if ($this->get_option('is_digital') === 'yes') {
                        $order->update_status('completed', $note);
                        $order->reduce_order_stock();
                        $order->payment_complete($transaction_id);
                    } else {
                        $order->update_status('processing', $note);
                        $order->reduce_order_stock();
                        $order->payment_complete($transaction_id);
                    }
                    return true;
                } else {
                    $status_message = isset($data['status']) ? $data['status'] : __('unknown', 'infinitypay');
                    $order->update_status('on-hold', sprintf(__('Infinity Pay payment status: %s. Please verify manually.', 'infinitypay'), $status_message));
                    return false;
                }
            }
            return true;
        }

        public function handle_webhook()
        {
            $order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;
            if (!$order_id) {
                status_header(400);
                wp_send_json_error('Order ID not provided');
                exit();
            }

            $order = wc_get_order($order_id);
            if (!$order) {
                status_header(404);
                wp_send_json_error('Order not found');
                exit();
            }

            $this->update_order_status($order);

            status_header(200);
            wp_send_json_success('Webhook processed successfully');
            exit();
        }
    }

    function infinitypay_add_gateway_class($gateways)
    {
        $gateways[] = 'WC_InfinityPay_Gateway';
        return $gateways;
    }
    add_filter('woocommerce_payment_gateways', 'infinitypay_add_gateway_class');
}

function infinitypay_handle_webhook()
{
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $transactionId = isset($_REQUEST['transactionId']) ? sanitize_text_field($_REQUEST['transactionId']) : '';
        if (empty($transactionId)) {
            status_header(400);
            wp_send_json_error('Transaction ID not provided');
            exit();
        }

        $gateway = new WC_InfinityPay_Gateway();
        $data = array(
            "transaction_id" => $transactionId,
        );
        
        $header = array(
            "api" => $gateway->get_option('apikeys'),
            "url" => rtrim($gateway->get_option('payment_site'), '/') . "/api/payment/verify"
        );

        $response = $gateway->create_payment($data, $header);
        $data = json_decode($response, true);

        if (isset($_GET['success1']) && isset($data['status']) && $data['status'] == "COMPLETED") {
            $order_id = intval($_GET['success1']);
            $order = wc_get_order($order_id);

            if ($order) {
                $order->update_status('completed', __('Payment confirmed via Infinity Pay webhook.', 'infinitypay'));
                $order->reduce_order_stock();
                $order->payment_complete($transactionId);
                
                // Empty cart
                WC()->cart->empty_cart();
            }
        }
    }

    status_header(200);
    wp_send_json_success('Webhook processed');
    exit();
}

add_action('rest_api_init', function () {
    register_rest_route('infinitypay/v1', '/webhook', array(
        'methods' => 'POST',
        'callback' => 'infinitypay_handle_webhook',
        'permission_callback' => '__return_true'
    ));
});